/* Credit Suisse SCORM 1.2 LMS functions for CS WBT Communication Standards v1.3 04-July-2012 */

/* assign CS-specific values to LMS variables */
lms_type = COMMUNICATION_INTERFACE;
if (lms_type == "NONE")
{
	lms_type = "none";
}
else if (lms_type == "SCORM")
{
	lms_type = "scorm12";
}
lms_num_tries = COMMUNICATION_NUMOFATTEMPT;
lms_save_interval = 3;
lms_pass_status1 = COMMUNICATION_STATUS_MASTERYSCOREPASSED;
lms_pass_status2 = COMMUNICATION_STATUS_MASTERYSCOREPASSED;

/* CS-specific SCORM 1.2 LMS functions */

var findAPITries = 0;
document.errorCheckLooping = false;
document.errorMessagePresent = false;
document.errorNotifyPresent = false;

function IntervalObject(){}
document.intervalObject = new IntervalObject();
document.lmsCallListener = new IntervalObject();
var intCount;

/*
	Function: actionsFindAPI
	Arguments: the current window
	Description: check to see if win contains the SCORM 1.2 API adaptor
	Returns: null if SCORM 1.2 API could not be located
			 a handle to SCORM 1.2 API adapter is returned if the API was located
*/
function actionsFindAPI(win)
{
   /* Check to see if the window (win) contains the SCORM 1.2 API
      if the window (win) does not contain the API and
     the window (win) has a parent window and the parent window  
     is not the same as the window (win)
   */ 
   while ( (win.API == null) && 
           (win.parent != null) && 
           (win.parent != win) )
   {
      /* increment the number of findAPITries */
      findAPITries++;

      /* note: 7 is an arbitrary number, but would be more than sufficient */
      if (findAPITries > lms_num_tries)
      {
         return null;
      }

      /* set the variable that represents the window being 
         being searched to be the parent of the current window
         then search for the API again
	  */	 
      win = win.parent;
   }
   return win.API;
}

/*
	Function: actionsGetAPI
	Arguments: none
	Description: locate the SCORM 1.2 API adaptor
	Returns: null if SCORM 1.2 API could not be located
			 a handle to API adapter is returned if the API was located
*/
function actionsGetAPI()
{
   /* start by looking for the SCORM 1.2 API in the current window */
   var theAPI = actionsFindAPI(window);
   /* if the API is null (could not be found in the current window)
	  and the current window has an opener window
   */ 
   if ( (theAPI == null) && 
        (window.opener != null) && 
        (typeof(window.opener) != "undefined") )
   {
      /* try to find the API in the current window's opener */
      theAPI = actionsFindAPI(window.opener);
   }
   /* if the API is null (could not be found in the current window,
      or the current windows opener) see if the opener of the topmost
      window contains the api
      and the current window has an opener window
   */	  
   if ( (theAPI == null) && 
        (window.top.opener != null) && 
        (typeof(window.top.opener) != "undefined") )
   {
      // try to find the API in the current window's opener
      theAPI = actionsFindAPI(window.top.opener);
   }
   return theAPI;
}

/*
	Function: actionsLMSInitialize
	Arguments: none		
	Description: initializes the LMS API
	Returns: boolean for success of initialization
*/
function actionsLMSInitialize()
{
	/* if LMS API adaptor exists and this is a SCORM 1.2 course */
	if(LMSAPI != null && lms_type == "scorm12"){
		/* attempt to initialize API */
		var result = LMSAPI.LMSInitialize("");
		/* retry if API not initialized */
		if(result==undefined){
			reTryLMSCall('LMSInitialize','','','01','WBT_startErrorInitTimeout')
		}
		/* try again as long as we haven't exceeded lms_num_tries */
		if(getLastError() != 0){
			reTryLMSCall('LMSInitialize','','','02','WBT_startErrorInitTimeout')
		}
		
	}
	return result;
}

/*
	Function: actionsLMSFinish
	Arguments: none		
	Description: executes final LMS calls on exit of the course
	Returns: nothing
*/
function actionsLMSFinish()
{
	/* check that the code has not already executed */
	if(!document.courseFinished)
	{
		/* if LMS API adaptor exists and this is a SCORM 1.2 course */
		if(LMSAPI != null && lms_type == "scorm12"){
			/* attempt to finish up with the LMS */
			var less_status = actionsLMSGetValue('cmi.core.lesson_status');
			/* if there is an error with the LMS communication report on it */
			var currentError = getLastError();
			if(currentError != 0) {
				actionsLMSError("LMSGetValue","cmi.core.lesson_status","startErrorGet",currentError);
			}				
			/* check status completion flag variable */
			if(less_status == lms_pass_status1){
				/* do nothing */
			}else{					
				actionsLMSSetValue('cmi.core.lesson_status',COMMUNICATION_STATUS_MASTERYSCOREFAILED,'001','wbtEnd')
				/* if there is an error with the LMS communication report on it */
				var currentError = getLastError();					
				if(currentError != 0) {
					actionsLMSError("LMSSetValue","cmi.core.lesson_status","startErrorSet",currentError);
				}				
			}
			/* set tracking data on closing course */
			recordPercentComplete(document.currentSection, document.subsectionCount, false, true);
			/* set the session time */
			courseWriteTimer(false);
			/* commit date to the LMS */
			
			//20210323 Edge Fix
			actionsLMSSetValue('cmi.core.exit',lms_exit_value,'068','WBT_startErrorSet');
			
			actionsLMSCommit();
			
			
			
			/* if there is an error with the LMS communication report on it */
			var currentError = getLastError();
			if(currentError != 0) {
				actionsLMSError("LMSCommit","","startErrorSet",currentError);
			}
			LMSAPI.LMSFinish('');
			document.courseFinished = true;
		}
	}
}

/*
	Function: actionsLMSCommit
	Arguments: none
	Description: commits the current value to the LMS
	Returns: nothing
*/
function actionsLMSCommit()
{
	/* if LMS API adaptor exists */
	if(LMSAPI != null){
		/* if not currently checking for an error in communicating with the LMS */
		if(document.errorCheckLooping == false){
			LMSAPI.LMSCommit('');
			/* if committing after setting complete or passed lesson status or setting a passing and higher score than LMS score use a more detailed error message */
			if (document.setCourseComplete) {
				handleError('LMSCommit','','','5050','WBT_endErrorSet_Data_Score',503,true);				
				document.setCourseComplete = false;
			}
			else if (document.setPassingScore2) {
				handleError('LMSCommit','','','5050','WBT_endErrorSet_Data_Score',503,true);
				document.setPassingScore2 = false;
			}
			else {			
				handleError('LMSCommit','','','5050','WBT_startErrorSet',500,true);
			}
		}else{
			/* add this to a queue to be called after the current LMS communication attempts are complete */
			addToCallQueue('LMSCommit','','','5050','WBT_startErrorSet',500,true)
		}
	}
}

/*
	Function: actionsLMSSetValue
	Arguments: 
		arg: database field e.g. cmi.core.session_time
		val: value for the database field
		id: an identifier for the call
		errorMsg: error message to use if required
	Description: sets the database field value in the LMS and commits it to the LMS		 
	Returns: nothing
*/
function actionsLMSSetValue(arg,val,id,errormsg)
{
	/* if LMS API adaptor exists */
	if(LMSAPI != null){
		/* if not currently checking for an error in communicating with the LMS */
		if(document.errorCheckLooping == false){
			id = val+"id";
			var valFromLMS = String(LMSAPI.LMSSetValue(arg,val));
			handleError('LMSSetValue',arg,val,id,errormsg);
		}else{
			/* add this to a queue to be called after the current LMS communication attempts are complete */
			id = val+"id"
			addToCallQueue('LMSSetValue',arg,val,id,errormsg)
		}
	}	
	/* ensure value is updated in LMS on tablet or phone but not in the Saba LMS app */
	if(!isWeb){
		if(androidApp == false && iOSApp == false){
			actionsLMSCommit();
		}
		if(androidApp == true){
			actionsLMSCommit(); //20210324 Android Fix
		}
	}else{
		actionsLMSCommit(); //20210323 Edge Fix
	}
}

/*
	Function: actionsLMSGetValue
	Arguments: 
		arg: database field e.g. cmi.core.session_time
	Description: obtains the database field value from the LMS				 
	Returns: value of the database field
*/
function actionsLMSGetValue(arg)
{
	var result = "";
	/* if LMS API adaptor exists */
	if(LMSAPI != null){
		/* if not currently checking for an error in communicating with the LMS */
		if(document.errorCheckLooping == false){
			var valFromLMS = String(LMSAPI.LMSGetValue(arg));
			id = arg+"id"
			handleError('LMSGetValue',arg,'',id,'WBT_startErrorGet',502,true);
				if(valFromLMS.search(/^\s*$/) != -1){
					/* leave the value as nothing */
				}else{
					result = valFromLMS;				
				};
		}else{
			/* add this to a queue to be called after the current LMS communication attempts are complete */
			id = arg+"id"
			addToCallQueue('LMSGetValue',arg,'',id,'WBT_startErrorGet',502,true)
		}
	}
	return result;
}

/*
	Function: actionsLMSError
	Arguments: 
		errLMS: code to indicate the type of LMS call that was attempted
		errArg: database field e.g. cmi.core.session_time or code for a database field		
		errType: error message code
		errCode: LMS error code
	Description: if configured to show error popups populates and shows the error popup
				 as long as there has not been a previous unresolved error on the score
				 screen as this would have been already reported
	Returns: none
*/
function actionsLMSError(errLMS,errArg,errType,errCode)
{	
	/* always show extended error message if required, even if error messages are turned off */
	if (errType == "endErrorSet_Data_Score")
	{
		var errorMessageTitle = getErrorString("WBT_errorTitle");
		var errorString = getErrorString("WBT_"+errType);
		var errorString2 = getErrorMessageDateTime();
		var errorString3 = getErrorString("WBT_"+errType+"2");
		errorString2 = getErrorString("WBT_"+errType+"2") + " " + getErrorMessageErrorCode(errCode);
		errorString3 = getErrorMessageWBTTitle() + "<br>" + getErrorMessageDateTime() + "<br>" + getErrorMessageUser() + "<br>" + getErrorMessageScore() + "<br>" + getErrorMessageStatus();
		showErrorMessage(errorMessageTitle, errorString, errorString2, errorString3);
	}
	else
	{	
		if(COMMUNICATION_ERRORMESSAGE){
			var errorMessageTitle = getErrorString("WBT_errorTitle");
			var errorString = getErrorString("WBT_"+errType);
			var errorString2 = getErrorMessageDateTime();
			var errorString3 = getErrorString("WBT_"+errType+"2");
			if ((errArg == "studentId") || (errArg == "studentName") || (errCode !== ""))
			{
				errorString3 = errorString3 + " " + getErrorMessageErrorCode(errCode);
			}
			showErrorMessage(errorMessageTitle, errorString, errorString2, errorString3);			
		}
	}	
}

/*
	Function: actionsLMSSetScore
	Arguments: integer for score
	Description: calls function to set the score in the LMS
	Returns: none
*/
function actionsLMSSetScore(n)
{
	/* if setting passing and higher score than LMS use a more detailed error message */
	if (document.setPassingScore1)	{
		actionsLMSSetValue('cmi.core.score.raw',n,'063','WBT_endErrorSet_Data_Score');
		/* 20210422 Android and Edge Fixes: if committing after setting complete or passed lesson status or setting a passing and higher score than LMS score use a more detailed error message */
		document.setPassingScore1 = false;
	}
	else {
		actionsLMSSetValue('cmi.core.score.raw',n,'063','WBT_startErrorSet');
	}
}

/*
	Function: actionsLMSGetScore
	Arguments: none
	Description: calls function to get the score from the LMS
	Returns: integer for score
*/
function actionsLMSGetScore()
{
	return actionsLMSGetValue('cmi.core.score.raw');
}

/*
	Function: actionsLMSSetSuspendData
	Arguments: string for suspend data
	Description: calls function to set the suspend data in the LMS
	Returns: none
*/
function actionsLMSSetSuspendData(n) 
{
	actionsLMSSetValue('cmi.suspend_data',n,'201','WBT_startErrorSet');
}

/*
	Function: actionsLMSGetSuspendData
	Arguments: none
	Description: calls function to get the suspend data from the LMS
	Returns: string for suspend data
*/
function actionsLMSGetSuspendData() 
{
	return actionsLMSGetValue('cmi.suspend_data');
}

/*
	Function: actionsLMSSetLessonLocation
	Arguments: string for lesson location
	Description: calls function to set the lesson location in the LMS
	Returns: none
*/
function actionsLMSSetLessonLocation()
{
	actionsLMSSetValue('cmi.core.lesson_location',int_tracking.int_lesson_location,'21','WBT_startErrorSet');	
}

/*
	Function: actionsLMSGetLessonLocation
	Arguments: none
	Description: calls function to get the lesson location from the LMS
	Returns: string for lesson location
*/
function actionsLMSGetLessonLocation() 
{
	return actionsLMSGetValue('cmi.core.lesson_location');	
}

/*
	Function: actionsLMSSetLessonStatus
	Arguments: 
		m: string for lesson status to set
		n: boolean to indicate if lesson status should be set in the LMS
	Description: calls function to set the lesson status in a local variable
				 and/or in the LMS
	Returns: none
*/
function actionsLMSSetLessonStatus(m,n)
{
	/* if the lesson status is incomplete then set it in the LMS */
	if (m == "incomplete")
	{
		actionsLMSSetValue('cmi.core.lesson_status',COMMUNICATION_STATUS_MASTERYSCOREFAILED,'11','WBT_startErrorSet');	
		int_tracking.int_lesson_status = COMMUNICATION_STATUS_MASTERYSCOREFAILED;
	}
	/* if the lesson status is completed then set it in the LMS only if indicated by n */
	else if (m == "completed")
	{
		if (n == true)
		{
			/* if setting complete or passed lesson status use a more detailed error message */			
			actionsLMSSetValue('cmi.core.lesson_status',lms_pass_status1,'065','WBT_endErrorSet_Data_Score');
			int_tracking.int_lesson_status = lms_pass_status1;
			/* 20210422 Android and Edge Fixes: if committing after setting complete or passed lesson status or setting a passing and higher score than LMS score use a more detailed error message */
		}
		else
		{
			int_tracking.int_lesson_status = lms_pass_status1;
		}	
	}
}

/*
	Function: actionsLMSGetLessonStatus
	Arguments: none
	Description: calls function to get the lesson status from the LMS
	Returns: string for lesson status
*/
function actionsLMSGetLessonStatus()
{
	return actionsLMSGetValue('cmi.core.lesson_status');
}

/*
	Function: actionsLMSSetSessionTime
	Arguments: CMITimeSpan string
	Description: calls function to set the session time in the LMS
	Returns: none
*/
function actionsLMSSetSessionTime(n) 
{
	actionsLMSSetValue('cmi.core.session_time',n,'067','WBT_startErrorSet');
}

/*
	Function: actionsLMSGetStudentId
	Arguments: none
	Description: calls function to get the student id from the LMS
	Returns: a string of the student id
*/
function actionsLMSGetStudentId() 
{
	return actionsLMSGetValue('cmi.core.student_id');
}

/*
	Function: actionsLMSGetStudentName
	Arguments: none
	Description: calls function to get the student name from the LMS
	Returns: a string of the student name
*/
function actionsLMSGetStudentName() 
{
	return actionsLMSGetValue('cmi.core.student_name');
}

/* CS-specific LMS functions */

/*
	Function: reTryLMScall
	Arguments: 
		callType: LMS call e.g. LMSSetValue, LMSGetValue etc.
		param: database field e.g. cmi.core.session_time
		value: data
		id: an identifier for the call
		errorMsg: error message to use if required
	Description: is called by an error handling interval
				 adjusts interface display
				 re-attempts the LMS call
	Returns: nothing
*/
function reTryLMSCall(callType,param,value,id,errorMsg){ 
	displayOnScreenNotification(callType,param,value,id,errorMsg)
	var intId = "id"+id;
	var msec = Number(COMMUNICATION_TIMEOUT * 1000);
	clearInterval(document.intervalObject[intId]);
	document.intervalObject[intId] = setInterval ("doLMSCall('"+callType+"','"+param+"','"+value+"','"+id+"','"+ errorMsg +"');",msec);
	document.errorCheckLooping =true;	
};

/*
	Function: doLMSCall
	Arguments: 
		callType: LMS call e.g. LMSSetValue, LMSGetValue etc.
		param: database field e.g. cmi.core.session_time
		value: data
		id: an identifier for the call
		errorMsg: error message to use if required
	Description: is called by reTryLMSCall()
				 establishes if the maximum attempts were made
				 depending on result, either displays error message and closes or makes LMS communication
	Returns: nothing
*/
function doLMSCall(callType,param,value,id,errorMsg){
	var intId = "id"+id;
	if(this["intCount"+id] == undefined){
		this["intCount"+id] = new Object();
		this["intCount"+id].count = new Object();
		this["intCount"+id].id = new Object();
		this["intCount"+id].count = 1;
		this["intCount"+id].id = id;
	}else{
		this["intCount"+id].count += 1;
	}	
	if(this["intCount"+id].count >= lms_num_tries){
		clearInterval(document.intervalObject[intId]);
		this["intCount"+id].count = 0;
		intCount = undefined;
		/* we have tried to communicate as many times as permitted by lms_num_tries */
		/* if there is still an error report on it */
		var currentError = getLastError();					
		if(currentError != 0) {
			errorMsg = errorMsg.split("WBT_")[1];
			actionsLMSError(callType,param,errorMsg,currentError);
		}
		displayOnScreenNotification(callType,param,value,id,errorMsg,"clear")
		document.errorCheckLooping = false;
		/* call next item in queue */
		checkLmsCallQueue();
	}else{
		switch(callType){
			case "LMSInitialize":
			LMSAPI.LMSInitialize('');			
			break;
			case "LMSSetValue":
			LMSAPI.LMSSetValue(param,value);			
			break;
			case "LMSGetValue":
			LMSAPI.LMSGetValue(param);
			break;
			case "LMSCommit":
			LMSAPI.LMSCommit('');			
			break;
			case "LMSFinish":
			LMSAPI.LMSFinish('');			
			break;
		}
	}
}

/*
	Function: handleError
	Arguments: 
		lmscallback: LMS call e.g. LMSSetValue, LMSGetValue etc.
		param: database field e.g. cmi.core.session_time
		value: value for the database field
		id: an identifier for the call
		errormsg: error message to use if required
	Description: if there is an error with the LMS communication it tries the call again
	Returns: nothing
*/
function handleError(lmscallback,param,value,id,errormsg){
	/* if there is an error with the LMS communication */
	if(getLastError() != 0) {
		/* if not currently checking for an error in communicating with the LMS try the LMS call again */
		if(document.errorCheckLooping == false){
			 reTryLMSCall(lmscallback,param,value,id,errormsg);			
		}
	} 
}

/*
	Function: getLastError
	Arguments: none
	Description: obtains the error code from LMSGetLastError()
	Returns: the error code
*/
function getLastError()
{
	/* if LMS API adaptor exists */
	if(LMSAPI != null){
		return LMSAPI.LMSGetLastError();
	}	
}

/*
	Function: getErrorString
	Arguments: identifier - look up on the errorStrings object in error_strings.js	
	Description: obtains an error message string from JSON object		
	Returns: error message string or "Unknown Error"	
*/
function getErrorString(identifier){
	/* updated to use language to read correct error string */
	if(eval("document.errorStrings_" + document.selectedLang ).hasOwnProperty(identifier)){
		return eval("document.errorStrings_" + document.selectedLang )[identifier];
			
	}else{
		return "Unknown Error";
	}
}

/*
	Function: displayOnScreenNotification
	Arguments: 
		callType: LMS call e.g. LMSSetValue, LMSGetValue etc.
		param: database field e.g. cmi.core.session_time
		value: data
		id: an identifier for the call
		errorMsg: error message to use if required
		n: either nothing or string "clear"
	Description: sets the innerHTML of the HTML element "footerTextArea" 
				 is called by error handling code to display a message 
	Returns: nothing
*/
function displayOnScreenNotification(callType,param,value,id,errorMsg,n){
	/* always show extended error message if required, even if error messages are turned off */
	if(COMMUNICATION_ERRORMESSAGE || (errorMsg == "WBT_endErrorSet_Data_Score")){
		if ((n != "clear") && !document.errorMessagePresent)
		{
			boxClose();
		}	
		document.errorNotifyPresent = true;
		var interfaceItem = document.getElementById("footerTextArea");
		var preErrorPopupBg = document.getElementById("preErrorLBBg");	
		if(n=="clear"){
			preErrorPopupBg.style.visibility = "visible";
			interfaceItem.innerHTML = "";
			interfaceItem.style.backgroundColor = "";
		}else{
			if (!document.errorMessagePresent) 
			{
				preErrorPopupBg.style.visibility = "visible";
				interfaceItem.innerHTML = getErrorString("WBT_attemptComm");
				interfaceItem.style.backgroundColor = "#ffffff";
			}	
		}
	}	
}

/*
	Function: makeLMSCommunication
	Arguments:
		callType: LMS call e.g. LMSSetValue, LMSGetValue etc.
		param: database field e.g. cmi.core.session_time
		value: value for the database field
		id: an identifier for the call
		errormsg: error message to use if required
	Description: determines if it's safe to make LMS communication
				 if not, adds call to the call queue
				 if so makes the LMS communication
	Returns: nothing
*/
function makeLMSCommunication(callType,param,value,id,errormsg){	
	/* if not currently checking for an error in communicating with the LMS */
	if(document.errorCheckLooping == true){
		/* add this call to the queue and break */
		addToCallQueue(callType,param,value,id,errormsg);		
	}else{		
		/* if not make call */
		switch(callType){
			case "LMSInitialize":
			LMSAPI.LMSInitialize('');			
			handleError(callType,param,value,id,errormsg);
			break;
			case "LMSSetValue":
			LMSAPI.LMSSetValue(param,value);			
			handleError(callType,param,value,id,errormsg);
			/* call next item in queue */
			if(document.callQueue.length>0){
				checkLmsCallQueue();
			}
			break;
			case "LMSGetValue":
			LMSAPI.LMSGetValue(param);						
			handleError(callType,param,value,id,errormsg);
			/* call next item in queue */
			if(document.callQueue.length>0){
				checkLmsCallQueue();
			}
			break;
			case "LMSCommit":
			LMSAPI.LMSCommit('');			
			handleError(callType,param,value,id,errormsg);
			/* call next item in queue */
			if(document.callQueue.length>0){
				checkLmsCallQueue();
			}
			break;
			case "LMSFinish":
			LMSAPI.LMSFinish('');			
			handleError(callType,param,value,id,errormsg);			
			break;
		}
	}
}

/*
	Function: callObject
	Arguments: 
		lmscallback: LMS call e.g. LMSSetValue, LMSGetValue etc.
		param: database field e.g. cmi.core.session_time
		value: data
		id: an identifier for the call
		errormsg: error message to use if required
	Description: constructor, creates a callObject, which can be added to the call queue
	Returns: nothing
*/
function callObject(lmscallback,param,value,id,errormsg){
	this.lmscallback = lmscallback;
	this.param	=	param;
	this.value = value;
	this.id = id;
	this.errormsg = errormsg;
}

/*
	Function: addToCallQueue
	Arguments: 
		lmscallback: LMS call e.g. LMSSetValue, LMSGetValue etc.
		param: database field e.g. cmi.core.session_time
		value: data
		id: an identifier for the call
		errormsg: error message to use if required
	Description: adds the call to a queue of calls
	Returns: nothing
*/
function addToCallQueue(lmscallback,param,value,id,errormsg){
		/* if not already in queue add it */
		if(document.callQueue == null || document.callQueue == undefined){
			document.callQueue = new Array();
		}
		this.callObj = new callObject(lmscallback,param,value,id,errormsg)
		document.callQueue.push(this.callObj)		
		var tempString = "";
		for(var z=0; z<=document.callQueue.length-1; z++){
			tempString += "<br>"+ document.callQueue[z].lmscallback + " (" +document.callQueue[z].param+")";
		}		
}

/*
	Function: checkLmsCallQueue
	Arguments: intId - call queue id
	Description: checks if there are any items in the call queue id so calls the first item
				 then removes the item from the Array
	Returns: nothing
*/
function checkLmsCallQueue(intId){	
	if(document.callQueue != undefined || document.callQueue != null){
		/* if there is something in the queue */
		if(document.callQueue.length>0){			
			/* if not currently checking for an error in communicating with the LMS */
			if(document.errorCheckLooping == false){
				var obj = document.callQueue[0];
				document.callQueue.shift();
				makeLMSCommunication(obj.lmscallback,obj.param,obj.value,obj.id,obj.errormsg);
			}
		}else{
			/* nothing in the queue - clearInterval */
			clearInterval(document.lmsCallListener[intId]);
		}
	}else{
		/* no queue - clearInterval */
		clearInterval(document.lmsCallListener[intId]);
	}
}

/*
	Function: startLmsCallListener
	Arguments: intId - call queue id
	Description: clears and sets the interval for the LMS call queue
	Returns: nothing
*/
function startLmsCallListener(intId) {
		clearInterval(document.lmsCallListener[intId])
		document.lmsCallListener[intId] = setInterval("checkLmsCallQueue('"+intId+"')", 3000);
}

/*
	Function: getErrorMessageDateTime
	Arguments: none
	Description: creates a string of the current date and time used in an LMS error message
	Returns: string of date and time	
*/
function getErrorMessageDateTime() {
	var timeSuffix;
	var currentTime = new Date();
	var month = currentTime.getMonth() + 1;
	var day = currentTime.getDate();
	var year = currentTime.getFullYear();
	var hours = currentTime.getHours();
	var minutes = currentTime.getMinutes();
	if (minutes < 10) minutes = "0" + minutes;
	(hours > 11)? timeSuffix = "PM" : timeSuffix = "AM";	
	var WBT_endErrorSet_Date = getErrorString("WBT_endErrorSet_Date");
	WBT_endErrorSet_Date = WBT_endErrorSet_Date.split("[date]");
	WBT_endErrorSet_Date_display = WBT_endErrorSet_Date[0]+month+"/"+day+"/"+year +" "+WBT_endErrorSet_Date[1]+" / ";
	WBT_endErrorSet_Date_display = WBT_endErrorSet_Date_display.split("[time]");
	WBT_endErrorSet_Date_display = WBT_endErrorSet_Date_display[0]+" "+ hours+":"+minutes+" "+timeSuffix;
	return WBT_endErrorSet_Date_display;
}

/*
	Function: getErrorMessageWBTTitle
	Arguments: none
	Description: creates a string of the course title used in an LMS error message
	Returns: string of course title	
*/
function getErrorMessageWBTTitle() {
	var WBT_endErrorSet_wbtname = document.courseTitle;
	var WBT_endErrorSet_wbtname2 = getErrorString("WBT_endErrorSet_wbtname");
	WBT_endErrorSet_wbtname2 = WBT_endErrorSet_wbtname2.split("[wbtname]");
	WBT_endErrorSet_wbtname_display = WBT_endErrorSet_wbtname2[0]+" "+ WBT_endErrorSet_wbtname +" "+WBT_endErrorSet_wbtname2[1];
	return WBT_endErrorSet_wbtname_display;
}

/*
	Function: getErrorMessageUser
	Arguments: none
	Description: creates a string of the student name and id used in an LMS error message
	Returns: string of student name and id
*/
function getErrorMessageUser() {
	var curStudentName = document.userName;
	var curStudentId = document.userId;	
	var WBT_endErrorSet_User = getErrorString("WBT_endErrorSet_User");
	WBT_endErrorSet_User_display = WBT_endErrorSet_User.split("[student_name]");
	WBT_endErrorSet_User_display = WBT_endErrorSet_User_display[0]+" "+ curStudentName+" "+WBT_endErrorSet_User_display[1];
	WBT_endErrorSet_User_display = WBT_endErrorSet_User_display.split("[student_id]");
	WBT_endErrorSet_User_display = WBT_endErrorSet_User_display[0]+" "+ curStudentId +" "+WBT_endErrorSet_User_display[1];
	return WBT_endErrorSet_User_display;
}

/*
	Function: getErrorMessageScore
	Arguments: none
	Description: creates a string of the current test score used in an LMS error message
	Returns: string of current test score
*/
function getErrorMessageScore() {
	var curScore = document.assessPercentageScore;
	var WBT_endErrorSet_Score = getErrorString("WBT_endErrorSet_Score");
	WBT_endErrorSet_Score_display = WBT_endErrorSet_Score.split("[score]");
	WBT_endErrorSet_Score_display = WBT_endErrorSet_Score_display[0]+" "+ curScore +" "+WBT_endErrorSet_Score_display[1];
	return WBT_endErrorSet_Score_display;
}

/*
	Function: getErrorMessageStatus
	Arguments: none
	Description: creates a string of the current lesson status used in an LMS error message
	Returns: string of current lesson status
*/
function getErrorMessageStatus() {
	var WBT_endErrorSet_Status = getErrorString("WBT_endErrorSet_Status");	
	WBT_endErrorSet_Status_display = WBT_endErrorSet_Status.split("[status]");
	WBT_endErrorSet_Status_display = WBT_endErrorSet_Status_display[0]+" "+int_tracking.int_lesson_status+" "+WBT_endErrorSet_Status_display[1];
	return WBT_endErrorSet_Status_display;
}

/*
	Function: getErrorMessageErrorCode
	Arguments: curErrorCode - current LMS error code
	Description: creates a string of the current LMS error code used in an LMS error message
	Returns: string of current LMS error code
*/
function getErrorMessageErrorCode(curErrorCode) {
	var curErrorCode = getLastError();
	var WBT_endErrorSet_ErrorCode = getErrorString("WBT_endErrorSet_ErrorCode");
	WBT_endErrorSet_ErrorCode_display = WBT_endErrorSet_ErrorCode.split("[errorcode]");
	WBT_endErrorSet_ErrorCode_display = WBT_endErrorSet_ErrorCode_display[0]+" "+curErrorCode+" "+WBT_endErrorSet_ErrorCode_display[1];;
	return WBT_endErrorSet_ErrorCode_display;
}